using UnityEngine;
using UnityEngine.SceneManagement;
using TMPro;
using System.Collections;

public class ScreenController : MonoBehaviour
{
    [SerializeField] private GameObject gameScreen;
    [SerializeField] private GameObject characters;
    [SerializeField] private GameObject mainScreen; 
    [SerializeField] private GameObject gameOverScreen;
    [SerializeField] private GameObject signUpScreen;
    [SerializeField] private GameObject rulesScreen;
    [SerializeField] private GameObject thankYouScreen;
    [SerializeField] private GameObject gameBackgrounds;
    [SerializeField] private TMP_Text scoreText; 
    [SerializeField] private bool setOnStart = true;
    [SerializeField] private ScreenTransitionsPro.ScreenTransitionDisplacementSoft displacementEffect;
    [SerializeField] private API api;
    private GameObject core;
    private ScoreBoard scoreBoard;
    private float timePassed = 0f;
    private bool signUpScreenIsActive = false;
    public bool SignUpScreenIsActive { get { return signUpScreenIsActive; } }

    private void Start()
    {
        SetOnStart();
        Globals.FadeTime = 0.5f;
        core = this.gameObject;
        scoreBoard = core.GetComponent<ScoreBoard>();
    }

    private void SetOnStart()
    {
        if (!setOnStart) return;

        mainScreen.SetActive(true);
        rulesScreen.SetActive(false);
        gameOverScreen.SetActive(false);
        signUpScreen.SetActive(false);
        gameScreen.SetActive(false);
        thankYouScreen.SetActive(false);
        gameBackgrounds.SetActive(false);
    }

    private void Update()
    {
        CheckMouseButton();
    }

    private void CheckMouseButton()
    {
        if (Input.GetMouseButtonDown(0) && mainScreen.activeSelf)
        {
            if (Globals.UseEffects)
            {
                StartTransition(GoToGame);
            }
            else
            {
                StartFade(mainScreen, gameScreen);
                api.PostStart();
            }
        }
    }

    private void StartTransition(UnityEngine.Events.UnityAction call)
    {
        displacementEffect.transitioning = true;
        displacementEffect.transitionEvent.AddListener(call);
    }

    public void GoToGame()
    {
        gameScreen.SetActive(true);
        characters.SetActive(true);
        mainScreen.gameObject.SetActive(false);
        displacementEffect.transitionEvent.RemoveListener(GoToGame);
        displacementEffect.transitionEvent.AddListener(HandleGameOver);
        api.PostStart();
    }

    private void GameOver()
    {
        gameOverScreen.SetActive(true);
        scoreText.text = scoreBoard.Score.ToString() + " BODOVA";
        displacementEffect.transitionEvent.RemoveListener(GameOver);
    }

    public void SetSignUpScreenOn()
    {
        if (Globals.UseEffects)
        {
            StartTransition(ActivateSignup);
        }
        else
        {
            StartFade(gameOverScreen, signUpScreen);
            signUpScreenIsActive = true;
        }
    }

    public void ActivateSignup()
    {
        rulesScreen.SetActive(false);
        signUpScreen.SetActive(true);
        signUpScreenIsActive = true;
    }

    public void HandleGameOver()
    {
        if (Globals.UseEffects)
        {
            StartTransition(GameOver);
        }
        else
        {
            gameBackgrounds.SetActive(true);
            StartFade(gameScreen, gameOverScreen);
            scoreText.text = scoreBoard.Score.ToString() + " BODOVA";
        }
    }

    public void SetRulesScreenOn()
    {
        if (Globals.UseEffects)
        {
            StartTransition(ActivateRules);
        }
        else
        {
            StartFade(signUpScreen, rulesScreen);
        }
    }

    public void DisableRules()
    {
        if (Globals.UseEffects)
        {
            StartTransition(ActivateSignup);
        }
        else
        {
            StartFade(rulesScreen, signUpScreen);
        }
    }

    public void SetThankYouScreenOn()
    {
        if (Globals.UseEffects)
        {
            StartTransition(ActivateThankYouScreen);
        }
        else
        {
            StartFade(signUpScreen, thankYouScreen);
        }
    }

    public void ActivateRules()
    {
        signUpScreen.SetActive(false);
        rulesScreen.SetActive(true);
    }

    public void ActivateThankYouScreen()
    {
        signUpScreen.SetActive(false);
        thankYouScreen.SetActive(true);
    }

    private void StartFade(GameObject currentScreen, GameObject nextScreen)
    {
        currentScreen.transform.SetSiblingIndex(1);
        nextScreen.transform.SetSiblingIndex(0);
        nextScreen.SetActive(true);
        nextScreen.GetComponent<CanvasGroup>().alpha = 1;
        StartCoroutine(Fade(currentScreen, nextScreen));
    }

    private IEnumerator Fade(GameObject currentScreen, GameObject nextScreen)
    {
        CanvasGroup canvasGroup = currentScreen.GetComponent<CanvasGroup>();
        canvasGroup.alpha = Mathf.Lerp(1, 0, timePassed / Globals.FadeTime);
        timePassed += Time.deltaTime;
        yield return new WaitForEndOfFrame();
        if (canvasGroup.alpha != 0)
        {
            StartCoroutine(Fade(currentScreen, nextScreen));
        }
        else
        {
            currentScreen.SetActive(false);
            timePassed = 0f;
        }
    }

    public void ReloadGame()
    {
        SceneManager.LoadScene(SceneManager.GetActiveScene().buildIndex);
    }
}