import sys
import cv2
import mediapipe as mp
import numpy as np

# Be careful not to write debug stuff into stdout

mp_pose = mp.solutions.pose


def main():
    print("Started mediapipe python pose segmentation", file=sys.stderr)
    input = sys.stdin.buffer
    output = sys.stdout.buffer

    with mp_pose.Pose(
        model_complexity=1, enable_segmentation=True, min_detection_confidence=0.5
    ) as pose:
        while True:
            frame = read_int_or_exit(input)
            length = read_int_or_exit(input)
            stride = read_int_or_exit(input)
            width = read_int_or_exit(input)
            top = read_int_or_exit(input)
            left = read_int_or_exit(input)
            height = int(length / stride)
            buf = input.read(length)

            if len(buf) != length:
                print(f"Read {len(buf)}/{length} bytes, exiting", file=sys.stderr)
                sys.exit(1)

            raw_img = np.frombuffer(buf, dtype=np.uint8).reshape(
                (height, int(stride / 4), 4)
            )
            img = cv2.cvtColor(raw_img, cv2.COLOR_RGBA2BGR)
            results = pose.process(img)

            if results.segmentation_mask is not None and results.pose_landmarks:
                raw_bytes = results.segmentation_mask.tobytes()
                landmarks = results.pose_landmarks.SerializeToString()
                landmarks_size = len(landmarks)

                to_le_bytes = lambda x: x.to_bytes(4, "little")
                header = b"".join(
                    list(
                        map(
                            to_le_bytes,
                            [
                                frame,
                                width,
                                height,
                                stride,
                                top,
                                left,
                                landmarks_size,
                            ],
                        )
                    )
                )
                size = len(header) + len(raw_bytes) + len(landmarks)
                data = b"".join(
                    [size.to_bytes(4, "little"), header, raw_bytes, landmarks]
                )
                output.write(data)
                output.flush()
            else:
                zero = 0
                output.write(zero.to_bytes(4, "little"))
                output.flush()


def read_int_or_exit(input):
    buf = input.read(4)
    if len(buf) != 4:
        sys.exit(1)
    return int.from_bytes(buf, byteorder="little")


if __name__ == "__main__":
    import argparse

    parser = argparse.ArgumentParser()
    # TODO ability to specify model
    parser.add_argument("--model", action="store_true")
    args = parser.parse_args()

    main()
